// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "include/base/cef_weak_ptr.h"

namespace base::cef_internal {

WeakReference::Flag::Flag() {
  // Flags only become bound when checked for validity, or invalidated,
  // so that we can check that later validity/invalidation operations on
  // the same Flag take place on the same thread.
  thread_checker_.DetachFromThread();
}

void WeakReference::Flag::Invalidate() {
  // The flag being invalidated with a single ref implies that there are no
  // weak pointers in existence. Allow deletion on other thread in this case.
#if DCHECK_IS_ON()
  DCHECK(thread_checker_.CalledOnValidThread() || HasOneRef())
      << "WeakPtrs must be invalidated on the same thread as where "
      << "they are bound.\n";
#endif
  invalidated_.Set();
}

bool WeakReference::Flag::IsValid() const {
  // WeakPtrs must be checked on the same thread.
  DCHECK(thread_checker_.CalledOnValidThread());
  return !invalidated_.IsSet();
}

bool WeakReference::Flag::MaybeValid() const {
  return !invalidated_.IsSet();
}

#if DCHECK_IS_ON()
void WeakReference::Flag::DetachFromThread() {
  thread_checker_.DetachFromThread();
}

void WeakReference::Flag::BindToCurrentThread() {
  thread_checker_.DetachFromThread();
  DCHECK(thread_checker_.CalledOnValidThread());
}
#endif

WeakReference::Flag::~Flag() = default;

WeakReference::WeakReference() = default;
WeakReference::WeakReference(const scoped_refptr<Flag>& flag) : flag_(flag) {}
WeakReference::~WeakReference() = default;

WeakReference::WeakReference(const WeakReference& other) = default;
WeakReference& WeakReference::operator=(const WeakReference& other) = default;

WeakReference::WeakReference(WeakReference&& other) noexcept = default;
WeakReference& WeakReference::operator=(WeakReference&& other) noexcept =
    default;

void WeakReference::Reset() {
  flag_ = nullptr;
}

bool WeakReference::IsValid() const {
  return flag_ && flag_->IsValid();
}

bool WeakReference::MaybeValid() const {
  return flag_ && flag_->MaybeValid();
}

WeakReferenceOwner::WeakReferenceOwner()
    : flag_(MakeRefCounted<WeakReference::Flag>()) {}

WeakReferenceOwner::~WeakReferenceOwner() {
  if (flag_) {
    flag_->Invalidate();
  }
}

WeakReference WeakReferenceOwner::GetRef() const {
#if DCHECK_IS_ON()
  DCHECK(flag_);
  // If we hold the last reference to the Flag then detach the ThreadChecker.
  if (!HasRefs()) {
    flag_->DetachFromThread();
  }
#endif

  return WeakReference(flag_);
}

void WeakReferenceOwner::Invalidate() {
  DCHECK(flag_);
  flag_->Invalidate();
  flag_ = MakeRefCounted<WeakReference::Flag>();
}

void WeakReferenceOwner::InvalidateAndDoom() {
  DCHECK(flag_);
  flag_->Invalidate();
  flag_.reset();
}

void WeakReferenceOwner::BindToCurrentThread() {
#if DCHECK_IS_ON()
  DCHECK(flag_);
  flag_->BindToCurrentThread();
#endif
}

WeakPtrFactoryBase::WeakPtrFactoryBase(uintptr_t ptr) : ptr_(ptr) {
  DCHECK(ptr_);
}

WeakPtrFactoryBase::~WeakPtrFactoryBase() {
  ptr_ = 0;
}

}  // namespace base::cef_internal
